using System;
using System.Runtime.InteropServices;

namespace Sony
{
	namespace NP
	{
		public class Ticketing
		{
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct Ticket
			{
				IntPtr _data;						// The ticket data.
				public int dataSize;				// The size of the ticket data.
				public byte[] data
				{
					get
					{
						byte[] bytes = new byte[dataSize];
						Marshal.Copy(_data, bytes, 0, dataSize);
						return bytes;
					}
				}
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TicketEntitlement
			{
				IntPtr _id;
				public int type;
				public int remainingCount;  // may be negative
				public int consumedCount;
				public UInt64 createdDate;
				public UInt64 expireDate;
				public string id { get { return Marshal.PtrToStringAnsi(_id); } }
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TicketEntitlementArray
			{
				IntPtr _data;						// The entitlement array data.
				public int count;					// The number of elements in the entilement array.
				public TicketEntitlement[] entitlements
				{
					get
					{
						TicketEntitlement[] ents = new TicketEntitlement[count];
						int structSize = Marshal.SizeOf(typeof(TicketEntitlement));
						for(int i=0; i<count ; i++)
						{
							IntPtr data = new IntPtr(_data.ToInt64() + structSize * i);
							ents[i] = (TicketEntitlement)Marshal.PtrToStructure(data, typeof(TicketEntitlement));
						}

						return ents;
					}
				}
			};

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TicketInfo
			{
				public long issuedDate;
				public long expireDate;
				public long subjectAccountID;
				public long statusDuration;
				int serialIDSize;
				IntPtr _serialID;
				IntPtr _subjectOnlineID;
				IntPtr _countryCode;
				IntPtr _subjectDomain;
				IntPtr _serviceID;
				public int issuerID;
				public int languageCode;
				public int subjectAge;
				public int chatDisabled;
				public int contentRating;

				public byte[] serialID
				{
					get
					{
						byte[] bytes = new byte[serialIDSize];
						Marshal.Copy(_serialID, bytes, 0, serialIDSize);
						return bytes;
					}
				}
				public string subjectOnlineID { get { return Marshal.PtrToStringAnsi(_subjectOnlineID); } }
				public string countryCode { get { return Marshal.PtrToStringAnsi(_countryCode); } }
				public string subjectDomain { get { return Marshal.PtrToStringAnsi(_subjectDomain); } }
				public string serviceID { get { return Marshal.PtrToStringAnsi(_serviceID); } }
			};


			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTicketingIsBusy();

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTicketingGetLastError(out ResultCode result);
			
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTicketingRequestTicket();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTicketingRequestCachedTicket();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTicketingGetTicket(out Ticket ticket);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTicketingGetTicketInfo(ref Ticket ticket, out TicketInfo info);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTicketingGetEntitlementList(ref Ticket ticket, out TicketEntitlementArray result);

			public static bool GetLastError(out ResultCode result)
			{
				PrxTicketingGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}
			
			// Is the word filter busy with a request.
			public static bool IsBusy()
			{
				return PrxTicketingIsBusy();
			}

			public static ErrorCode RequestTicket()
			{
				return PrxTicketingRequestTicket();
			}
			
			public static ErrorCode RequestCachedTicket()
			{
				return PrxTicketingRequestCachedTicket();
			}

			public static Ticket GetTicket()
			{
				Ticket ticket = new Ticket();
				PrxTicketingGetTicket(out ticket);
				return ticket;
			}

			public static TicketInfo GetTicketInfo(Ticket ticket)
			{
				TicketInfo info = new TicketInfo();
				PrxTicketingGetTicketInfo(ref ticket, out info);
				return info;
			}

			public static TicketEntitlement[] GetTicketEntitlements(Ticket ticket)
			{
				TicketEntitlementArray arrayData = new TicketEntitlementArray();
				PrxTicketingGetEntitlementList(ref ticket, out arrayData);
				TicketEntitlement[] ents = arrayData.entitlements;
				return ents;
			}

			// Event delegates.
			public static event Messages.EventHandler OnGotTicket;
			public static event Messages.EventHandler OnError;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_TicketingGotTicket:
						if (OnGotTicket != null) OnGotTicket(msg);
						return true;

					case Messages.MessageType.kNPToolKit_TicketingError:
						if (OnError != null) OnError(msg);
						return true;
				}
				
				return false;
			}
			
		}
	}
}
